<?php
/**
 * API SEM BANCO DE DADOS - USANDO ARQUIVOS JSON
 * Sistema de Status Online/Offline com arquivos
 */

// Configurar timezone para Brasil (Brasília)
date_default_timezone_set('America/Fortaleza'); // Ceará
// Ou use: 'America/Sao_Paulo' para SP/RJ
// Ou use: 'America/Manaus' para AM
// Ou use: 'America/Recife' para PE

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

// Diretório para armazenar dados
$dataDir = __DIR__ . '/data/';

// Criar diretório se não existir
if (!is_dir($dataDir)) {
    mkdir($dataDir, 0755, true);
}

// Arquivos de dados
$messagesFile = $dataDir . 'messages.json';
$sessionsFile = $dataDir . 'sessions.json';

// Pegar ação
$action = $_GET['action'] ?? $_POST['action'] ?? '';

switch($action) {
    case 'getMessages':
        getMessages($messagesFile, $sessionsFile);
        break;
    
    case 'sendMessage':
        sendMessage($messagesFile);
        break;
    
    case 'uploadMedia':
        uploadMedia($messagesFile);
        break;
    
    case 'clearConversation':
        clearConversation($messagesFile);
        break;
    
    case 'addReaction':
        addReaction($messagesFile);
        break;
    
    case 'fetchLinkPreview':
        fetchLinkPreview();
        break;
    
    case 'updateLastSeen':
        updateLastSeen($sessionsFile);
        break;
    
    case 'clearGallery':
        clearGallery($messagesFile);
        break;
    
    default:
        echo json_encode(['success' => false, 'error' => 'Ação inválida']);
}

/**
 * Carregar mensagens
 */
function getMessages($messagesFile, $sessionsFile) {
    // Carregar mensagens
    $messages = [];
    if (file_exists($messagesFile)) {
        $content = file_get_contents($messagesFile);
        $messages = json_decode($content, true) ?? [];
    }
    
    // Carregar última visualização do visitante
    $lastSeen = null;
    if (file_exists($sessionsFile)) {
        $content = file_get_contents($sessionsFile);
        $sessions = json_decode($content, true) ?? [];
        
        // Buscar sessão do visitante (user)
        foreach ($sessions as $session) {
            if ($session['user_type'] === 'user') {
                $lastSeen = $session['last_seen'];
                break;
            }
        }
    }
    
    echo json_encode([
        'success' => true,
        'messages' => $messages,
        'lastSeen' => $lastSeen
    ]);
}

/**
 * Enviar mensagem
 */
function sendMessage($messagesFile) {
    $sender = $_POST['sender'] ?? '';
    $type = $_POST['type'] ?? 'text';
    $content = $_POST['content'] ?? '';
    $replyTo = $_POST['replyTo'] ?? null;
    $linkPreview = $_POST['linkPreview'] ?? null;
    
    if (empty($sender) || empty($content)) {
        echo json_encode(['success' => false, 'error' => 'Dados incompletos']);
        return;
    }
    
    // Carregar mensagens existentes
    $messages = [];
    if (file_exists($messagesFile)) {
        $content_file = file_get_contents($messagesFile);
        $messages = json_decode($content_file, true) ?? [];
    }
    
    // Adicionar nova mensagem
    $newMessage = [
        'id' => uniqid(),
        'sender' => $sender,
        'type' => $type,
        'content' => $content,
        'timestamp' => date('Y-m-d H:i:s'),
        'liked' => false,
        'replyTo' => $replyTo
    ];
    
    // Adicionar linkPreview se existir
    if ($linkPreview) {
        $newMessage['linkPreview'] = json_decode($linkPreview, true);
    }
    
    $messages[] = $newMessage;
    
    // Salvar
    file_put_contents($messagesFile, json_encode($messages, JSON_PRETTY_PRINT));
    
    echo json_encode([
        'success' => true,
        'message' => $newMessage
    ]);
}

/**
 * Upload de mídia
 */
function uploadMedia($messagesFile) {
    $sender = $_POST['sender'] ?? '';
    $type = $_POST['type'] ?? '';
    
    if (empty($sender) || empty($type) || !isset($_FILES['file'])) {
        echo json_encode(['success' => false, 'error' => 'Dados incompletos']);
        return;
    }
    
    // Diretório para uploads
    $uploadDir = __DIR__ . '/uploads/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $file = $_FILES['file'];
    $fileName = uniqid() . '_' . basename($file['name']);
    $filePath = $uploadDir . $fileName;
    
    if (move_uploaded_file($file['tmp_name'], $filePath)) {
        // Carregar mensagens existentes
        $messages = [];
        if (file_exists($messagesFile)) {
            $content = file_get_contents($messagesFile);
            $messages = json_decode($content, true) ?? [];
        }
        
        // Adicionar mensagem com mídia
        $newMessage = [
            'id' => uniqid(),
            'sender' => $sender,
            'type' => $type,
            'content' => 'uploads/' . $fileName,
            'timestamp' => date('Y-m-d H:i:s'),
            'liked' => false
        ];
        
        $messages[] = $newMessage;
        
        // Salvar
        file_put_contents($messagesFile, json_encode($messages, JSON_PRETTY_PRINT));
        
        echo json_encode([
            'success' => true,
            'message' => $newMessage
        ]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Erro ao fazer upload']);
    }
}

/**
 * Limpar conversa
 */
function clearConversation($messagesFile) {
    // Criar arquivo vazio
    file_put_contents($messagesFile, json_encode([], JSON_PRETTY_PRINT));
    
    echo json_encode([
        'success' => true,
        'message' => 'Conversa limpa com sucesso'
    ]);
}

/**
 * Adicionar reação à mensagem
 */
function addReaction($messagesFile) {
    $messageId = $_POST['messageId'] ?? '';
    $reaction = $_POST['reaction'] ?? '';
    
    if (empty($messageId) || empty($reaction)) {
        echo json_encode(['success' => false, 'error' => 'Dados incompletos']);
        return;
    }
    
    // Carregar mensagens
    $messages = [];
    if (file_exists($messagesFile)) {
        $content = file_get_contents($messagesFile);
        $messages = json_decode($content, true) ?? [];
    }
    
    // Encontrar mensagem e adicionar/alternar reação
    $found = false;
    foreach ($messages as &$msg) {
        if ($msg['id'] === $messageId) {
            // Inicializar reactions se não existir
            if (!isset($msg['reactions'])) {
                $msg['reactions'] = [];
            }
            
            // Se já tem essa reação, remove (toggle)
            // Se não tem, adiciona
            if (in_array($reaction, $msg['reactions'])) {
                $msg['reactions'] = array_values(array_diff($msg['reactions'], [$reaction]));
            } else {
                $msg['reactions'][] = $reaction;
            }
            
            // Manter campo 'liked' para compatibilidade (se tiver coração)
            $msg['liked'] = in_array('❤️', $msg['reactions']);
            
            $found = true;
            break;
        }
    }
    
    if ($found) {
        // Salvar
        file_put_contents($messagesFile, json_encode($messages, JSON_PRETTY_PRINT));
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Mensagem não encontrada']);
    }
}

/**
 * Atualizar última visualização (status online)
 */
function updateLastSeen($sessionsFile) {
    session_start();
    
    $userType = $_POST['userType'] ?? 'user';
    $userId = session_id();
    $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    
    // Carregar sessões existentes
    $sessions = [];
    if (file_exists($sessionsFile)) {
        $content = file_get_contents($sessionsFile);
        $sessions = json_decode($content, true) ?? [];
    }
    
    // NOVA LÓGICA: Manter apenas 1 sessão por user_type
    // Remover todas as sessões antigas deste user_type
    $sessions = array_filter($sessions, function($session) use ($userType) {
        return $session['user_type'] !== $userType;
    });
    
    // Reindexar array (remover índices vazios)
    $sessions = array_values($sessions);
    
    // Adicionar/atualizar sessão atual
    $sessions[] = [
        'user_id' => $userId,
        'user_type' => $userType,
        'last_seen' => date('Y-m-d H:i:s'),
        'is_online' => true,
        'ip_address' => $ipAddress,
        'user_agent' => $userAgent,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    // Limpar sessões antigas (mais de 7 dias) - para outros user_types
    $sessions = array_filter($sessions, function($session) {
        $lastSeen = strtotime($session['last_seen']);
        $sevenDaysAgo = strtotime('-7 days');
        return $lastSeen > $sevenDaysAgo;
    });
    
    // Reindexar novamente
    $sessions = array_values($sessions);
    
    // Marcar como offline sessões inativas (mais de 5 minutos)
    foreach ($sessions as &$session) {
        $lastSeen = strtotime($session['last_seen']);
        $fiveMinutesAgo = strtotime('-5 minutes');
        if ($lastSeen < $fiveMinutesAgo) {
            $session['is_online'] = false;
        }
    }
    
    // Salvar
    file_put_contents($sessionsFile, json_encode($sessions, JSON_PRETTY_PRINT));
    
    echo json_encode([
        'success' => true,
        'timestamp' => date('Y-m-d H:i:s'),
        'user_id' => $userId
    ]);
}

/**
 * Função auxiliar: Limpar sessões antigas (chamar periodicamente)
 */
function cleanupSessions($sessionsFile) {
    if (!file_exists($sessionsFile)) return;
    
    $content = file_get_contents($sessionsFile);
    $sessions = json_decode($content, true) ?? [];
    
    // Remover sessões antigas
    $sessions = array_filter($sessions, function($session) {
        $lastSeen = strtotime($session['last_seen']);
        $sevenDaysAgo = strtotime('-7 days');
        return $lastSeen > $sevenDaysAgo;
    });
    
    // Marcar como offline sessões inativas
    foreach ($sessions as &$session) {
        $lastSeen = strtotime($session['last_seen']);
        $fiveMinutesAgo = strtotime('-5 minutes');
        if ($lastSeen < $fiveMinutesAgo) {
            $session['is_online'] = false;
        }
    }
    
    file_put_contents($sessionsFile, json_encode($sessions, JSON_PRETTY_PRINT));
}

/**
 * Limpar galeria (apagar apenas mensagens com mídia)
 */
function clearGallery($messagesFile) {
    try {
        // Carregar mensagens existentes
        $messages = [];
        if (file_exists($messagesFile)) {
            $content = file_get_contents($messagesFile);
            $messages = json_decode($content, true) ?? [];
        }
        
        // Contar mídias antes de apagar
        $mediaCount = 0;
        $uploadDir = __DIR__ . '/uploads/';
        
        // Filtrar mensagens: manter apenas mensagens de texto
        $filteredMessages = [];
        foreach ($messages as $msg) {
            if ($msg['type'] === 'text') {
                // Manter mensagens de texto
                $filteredMessages[] = $msg;
            } else {
                // É mídia, incrementar contador
                $mediaCount++;
                
                // Tentar deletar arquivo físico
                if (isset($msg['content'])) {
                    $filePath = __DIR__ . '/' . $msg['content'];
                    if (file_exists($filePath)) {
                        @unlink($filePath);
                    }
                }
            }
        }
        
        // Salvar apenas mensagens de texto
        file_put_contents($messagesFile, json_encode($filteredMessages, JSON_PRETTY_PRINT));
        
        echo json_encode([
            'success' => true,
            'message' => 'Galeria limpa com sucesso',
            'deleted_count' => $mediaCount,
            'remaining_messages' => count($filteredMessages)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
}

/**
 * Buscar preview de link (Open Graph)
 */
function fetchLinkPreview() {
    $url = $_POST['url'] ?? '';
    
    if (empty($url)) {
        echo json_encode(['success' => false, 'error' => 'URL não fornecida']);
        return;
    }
    
    // Validar URL
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        echo json_encode(['success' => false, 'error' => 'URL inválida']);
        return;
    }
    
    try {
        // Buscar conteúdo da página
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; LinkPreviewBot/1.0)');
        $html = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200 || !$html) {
            echo json_encode(['success' => false, 'error' => 'Erro ao buscar página']);
            return;
        }
        
        // Extrair meta tags Open Graph
        $title = '';
        $description = '';
        $image = '';
        
        // Title
        if (preg_match('/<meta property="og:title" content="([^"]+)"/i', $html, $match)) {
            $title = $match[1];
        } elseif (preg_match('/<title>([^<]+)<\/title>/i', $html, $match)) {
            $title = $match[1];
        }
        
        // Description
        if (preg_match('/<meta property="og:description" content="([^"]+)"/i', $html, $match)) {
            $description = $match[1];
        } elseif (preg_match('/<meta name="description" content="([^"]+)"/i', $html, $match)) {
            $description = $match[1];
        }
        
        // Image
        if (preg_match('/<meta property="og:image" content="([^"]+)"/i', $html, $match)) {
            $image = $match[1];
        }
        
        // Limitar tamanhos
        $title = mb_substr($title, 0, 100);
        $description = mb_substr($description, 0, 200);
        
        echo json_encode([
            'success' => true,
            'preview' => [
                'url' => $url,
                'title' => $title ?: parse_url($url, PHP_URL_HOST),
                'description' => $description,
                'image' => $image
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => 'Erro ao processar preview']);
    }
}

// Executar limpeza (opcional, mas recomendado)
// cleanupSessions($sessionsFile);

?>